import { defineStore } from 'pinia'
import type { Song } from '../types/song'

export const usePlayerStore = defineStore('player', {
  state: () => ({
    currentSong: null as Song | null,
    playlist: [] as Song[],
    isPlaying: false,
    currentTime: 0,
    volume: 1,
    audio: null as HTMLAudioElement | null,
  }),

  actions: {
    setPlaylist(songs: Song[]) {
      this.playlist = songs
    },

    async setCurrentSong(song: Song) {
      console.log('Setting current song:', song)
      this.currentSong = song
      const wasPlaying = this.isPlaying

      // 清理旧的音频实例
      if (this.audio) {
        this.audio.pause()
        this.audio.removeEventListener('ended', this.nextTrack)
        this.audio.removeEventListener('timeupdate', this.updateTime)
      }

      try {
        // 创建新的音频实例
        this.audio = new Audio()
        // 设置音频属性
        this.audio.preload = 'auto'
        this.audio.volume = this.volume

        // 添加事件监听
        this.audio.addEventListener('ended', this.nextTrack)
        this.audio.addEventListener('timeupdate', this.updateTime)
        this.audio.addEventListener('error', (e) => {
          console.error('Audio error:', e)
        })
        this.audio.addEventListener('canplay', () => {
          console.log('Audio can play')
          // 如果之前在播放，尝试自动播放
          if (wasPlaying) {
            this.play().catch((error) => {
              console.warn('Auto-play failed:', error)
              // 如果自动播放失败，重置播放状态
              this.isPlaying = false
            })
          }
        })
        this.audio.addEventListener('playing', () => {
          console.log('Audio is playing')
          this.isPlaying = true
        })
        this.audio.addEventListener('pause', () => {
          console.log('Audio paused')
          this.isPlaying = false
        })

        // 最后设置音频源
        this.audio.src = song.url
      } catch (error) {
        console.error('Error creating audio:', error)
      }
    },

    updateTime() {
      if (this.audio) {
        this.currentTime = this.audio.currentTime
      }
    },

    async play() {
      if (!this.audio) return

      try {
        // 预加载音频
        if (this.audio.readyState < 2) {
          console.log('Loading audio...')
          await new Promise((resolve) => {
            this.audio!.addEventListener('canplay', resolve, { once: true })
            this.audio!.load()
          })
        }
        await this.audio.play()
      } catch (error) {
        console.error('Error playing audio:', error)
      }
    },

    async pause() {
      if (!this.audio) return
      try {
        await this.audio.pause()
      } catch (error) {
        console.error('Error pausing audio:', error)
      }
    },

    async togglePlay() {
      console.log('Toggle play called, current state:', {
        audio: this.audio,
        isPlaying: this.isPlaying,
      })

      if (!this.audio) {
        console.warn('No audio instance available')
        return
      }

      try {
        if (this.isPlaying) {
          await this.pause()
        } else {
          await this.play()
        }
      } catch (error) {
        console.error('Error toggling playback:', error)
      }
    },

    async nextTrack() {
      if (!this.playlist.length) return
      const currentIndex = this.playlist.findIndex((song) => song.id === this.currentSong?.id)
      const nextIndex = (currentIndex + 1) % this.playlist.length
      await this.setCurrentSong(this.playlist[nextIndex])
    },

    async previousTrack() {
      if (!this.playlist.length) return
      const currentIndex = this.playlist.findIndex((song) => song.id === this.currentSong?.id)
      const prevIndex = (currentIndex - 1 + this.playlist.length) % this.playlist.length
      await this.setCurrentSong(this.playlist[prevIndex])
    },

    setVolume(volume: number) {
      this.volume = volume
      if (this.audio) {
        this.audio.volume = volume
      }
    },

    seek(time: number) {
      if (this.audio) {
        this.audio.currentTime = time
        this.currentTime = time
      }
    },
  },
})
